let peerConnection = null;
let localStream;
let inactivityTimer;
let lastActive = 0;
let ddl = 0;
const params = new URLSearchParams(window.location.search);
const audioval = params.get('value');
let deviceId = params.get('deviceId');

function resetInactivityTimer() {
    if (inactivityTimer) {
        clearTimeout(inactivityTimer);
    }

    inactivityTimer = setTimeout(() => {
        window.close(); 
    }, 60000); // 60 секунд
}

function setRemoteDescriptionSafely(sdpAnswer) {
    if (peerConnection.signalingState === 'have-local-offer') {
        peerConnection.setRemoteDescription(new RTCSessionDescription(sdpAnswer))
            .catch(error => console.error('Error setting remote description:', error));
    } else {
        peerConnection.addEventListener('signalingstatechange', function onStateChange() {
            if (peerConnection.signalingState === 'have-local-offer') {
                peerConnection.setRemoteDescription(new RTCSessionDescription(sdpAnswer))
                    .catch(error => console.error('Error setting remote description:', error));
                peerConnection.removeEventListener('signalingstatechange', onStateChange);
            }
        });
    }
}


function initializePeerConnection() {

    if (peerConnection !== null) {
        return;
    }
    
    peerConnection = new RTCPeerConnection();
    var candidates = [];

    peerConnection.onicecandidate = (event) => {
        if (event.candidate) {
            candidates.push(event.candidate);
        }
    };

    navigator.mediaDevices.getUserMedia({ audio: true, video: false })
        .then((stream) => {
            localStream = stream;
            stream.getTracks().forEach(track => peerConnection.addTrack(track, stream));
            return peerConnection.createOffer();
        })
        .then((offer) => peerConnection.setLocalDescription(offer))
        .then(() => {
            chrome.runtime.sendMessage({
                target: 'background',
                command: 'sendOffer',
                data: peerConnection.localDescription
            });
        })
        .catch((error) => console.warn('Error with WebRTC:', error));
}

function closeWRTC () {
    
    if (peerConnection) {
        peerConnection.close();
        peerConnection = null;
    }
    
    if (localStream) {
        localStream.getTracks().forEach(track => track.stop());        
    }
}

function playSound() {
    const audio = document.getElementById("ringing");
    audio.play().catch(error => console.error("Error playing sound:", error));
}

function playFail() {
    const audio = document.getElementById("fail");
    audio.play().catch(error => console.error("Error playing sound:", error));
}

function stopSound() {
    const audio = document.getElementById("ringing");
    audio.pause();
    audio.currentTime = 0;
}

async function checkMicrophonePermission() {
    try {
        const permissionStatus = await navigator.permissions.query({ name: 'microphone' });

        if (permissionStatus.state === 'denied') {
            alert("Microphone permission is denied. Please enable it in your browser settings.");
        } else if (permissionStatus.state === 'prompt') {

            try {
                await navigator.mediaDevices.getUserMedia({ audio: true });
            } catch (err) {
                console.error("Microphone permission denied.", err);
                alert("Microphone access denied.");
            }
        }
    } catch (err) {
        console.error("Permission API not supported", err);
    }
}

async function selectAudioOutput () {

    if (deviceId == 'undefined') return; // for default adioutput

    let audioElement = document.getElementById('ringing');
    let audioElement2 = document.getElementById('fail');

    if (audioElement) {

        if (typeof audioElement.setSinkId === 'function') {

            audioElement.setSinkId(deviceId)
                .catch((error) => {
                    console.error('Error setting sinkId:', error);
                });
        } else {
            console.error('setSinkId is not supported in this browser.', 'Browser version:', navigator.userAgent);
        }
    } else {
        console.error('Audio element not found.');
    }
    if (audioElement2) {

        if (typeof audioElement2.setSinkId === 'function') {

            audioElement2.setSinkId(deviceId)
                .catch((error) => {
                    console.error('Error setting sinkId:', error);
                });
        } else {
            console.error('setSinkId is not supported in this browser.', 'Browser version:', navigator.userAgent);
        }
    } else {
        console.error('Audio element not found.');
    }
}

window.addEventListener("load", checkMicrophonePermission);

resetInactivityTimer();
selectAudioOutput();

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {

    sendResponse({ success: true });
    if (message.target != audioval) return true;

    resetInactivityTimer();

    if (message.command === 'sendSDP') {
        initializePeerConnection();
    } else if (message.command === 'sdp' && peerConnection) {
        
        const signalData = message.data.msg_values;
        
        if (signalData.type === 'answer') {
            setRemoteDescriptionSafely(signalData);

            peerConnection.ontrack = (event) => {

                const [remoteStream] = event.streams;

                const audioElement = document.createElement('audio');

                audioElement.srcObject = remoteStream;

                audioElement.autoplay = true;

                document.body.appendChild(audioElement);

            };
        }

    } else if (message.command == 'closeWebRTC') {
        closeWRTC();
        stopSound();
    } else if (message.command == 'playSound') {
        playSound();
    } else if (message.command == 'stopSound') {
        stopSound();
    } else if (message.command == 'ping') {
	    chrome.storage.session.set({ pong: (lastActive = Math.floor(Date.now() / 1000)) });
    } else if (message.command == 'closeWindow') {
        window.close();
    } else if (message.command == 'selectAudioOutput') {
        deviceId = message.data.deviceId;
        selectAudioOutput();
    } else if (message.command == 'playFail') {
        playFail();
    }
    
    return true;

});

